/**
* \file: Referencable.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef CARPLAY_REFERENCABLE_H
#define CARPLAY_REFERENCABLE_H

#include "Common.h"
#include <map>
#include <typeinfo>
#include <pthread_adit.h>

namespace adit { namespace carplay
{

template <class TClass, typename TReference>
class CARPLAY_HIDDEN Referencable
{
public:
    Referencable() { reference = nullptr; }
    virtual ~Referencable()
    {
        // clean-up
        if (reference != nullptr)
        {
            pthread_mutex_lock(&mapLock);
            auto item = map.find(reference);
            if (item != map.end())
                map.erase(item);
            pthread_mutex_unlock(&mapLock);
        }
    }

    static TClass* Get(TReference ref);
    static TClass* Add(TReference ref);
    static void Remove(TReference ref);
    static bool Has(TReference ref);

    inline TReference GetReference() const { return reference; }

protected:
    TReference reference;

    inline size_t GetInstanceCount() const { return map.size(); }

private:
    static std::map<const TReference, TClass*> map;
    static pthread_mutex_t mapLock;
};

template <class TClass, typename TReference>
TClass* Referencable<TClass, TReference>::Get(TReference ref)
{
    TClass* ptr = nullptr;

    pthread_mutex_lock(&mapLock);
    auto item = map.find(ref);
    if (item != map.end())
        ptr = item->second;
    pthread_mutex_unlock(&mapLock);

#if 0
    if (ptr == nullptr)
    {
        item = map.find(nullptr);
        if (item != map.end())
        {
            LOG_WARN((dipo, "Referencable<%s>::Get(%p) = nullptr! Fall back to default.",
                    typeid(ref).name(), ref));
            return item->second;
        }
    }
#endif

    if (ptr == nullptr)
        LOG_ERROR((dipo, "Referencable<%s>::Get(%p) = nullptr!", typeid(ref).name(), ref));
    return ptr;
}

template <class TClass, typename TReference>
TClass* Referencable<TClass, TReference>::Add(TReference ref)
{
    TClass* ptr = nullptr;

    auto obj = new TClass();
    dipo_exit_on_null(obj);

    pthread_mutex_lock(&mapLock);
    auto item = map.find(ref);
    if (item == map.end())
    {
        obj->reference = ref;

        if (map.end() == map.find(nullptr))
            map[nullptr] = obj;
        ptr = (map[ref] = obj);
    }
    pthread_mutex_unlock(&mapLock);

    if (ptr == nullptr)
    {
        delete obj;
        LOG_ERROR((dipo, "Referencable<%s>::Add(%p) already added!", typeid(ref).name(), ref));
    }
    return ptr;
}

template <class TClass, typename TReference>
void Referencable<TClass, TReference>::Remove(TReference ref)
{
    TClass* ptr = nullptr;

    pthread_mutex_lock(&mapLock);
    auto def = map.find(nullptr);
    auto item = map.find(ref);
    if (item != map.end())
    {
        if (def != map.end() && def->second == item->second)
            map.erase(def);
        if (item->second != nullptr)
            item->second->reference = nullptr; // prevent removal in destructor
        ptr = item->second;
        map.erase(item);
    }
    pthread_mutex_unlock(&mapLock);

    if (ptr != nullptr)
        delete ptr;
}

template <class TClass, typename TReference>
bool Referencable<TClass, TReference>::Has(TReference ref)
{
    bool result;

    pthread_mutex_lock(&mapLock);
    auto item = map.find(ref);
    result = item != map.end();
    pthread_mutex_unlock(&mapLock);

    return result;
}

template <class TClass, typename TReference>
std::map<const TReference, TClass*> Referencable<TClass, TReference>::map;
template <class TClass, typename TReference>
pthread_mutex_t Referencable<TClass, TReference>::mapLock = PTHREAD_MUTEX_INITIALIZER;

} } /* namespace adit { namespace carplay */

#endif /* CARPLAY_REFERENCABLE_H */
